<?php

namespace App\Filament\Admin\Resources;

use Filament\Forms;
use Filament\Tables;
use App\Models\Client;
use Filament\Forms\Get;
use Filament\Forms\Set;
use App\Models\BrikUnit;
use Filament\Forms\Form;
use App\Models\BrikOwner;
use Filament\Tables\Table;
use Filament\Resources\Resource;
use Filament\Forms\Components\Hidden;
use Filament\Forms\Components\Select;
use Filament\Forms\Components\Toggle;
use Filament\Forms\Components\Section;
use Filament\Tables\Columns\TextColumn;
use Filament\Forms\Components\TextInput;
use Filament\Notifications\Notification;
use Filament\Forms\Components\DatePicker;
use Illuminate\Database\Eloquent\Builder;
use App\Actions\Utility\SelectDymanicNames;
use Filament\Forms\Components\MarkdownEditor;
use Illuminate\Database\Eloquent\SoftDeletingScope;
use App\Filament\Admin\Resources\BrikOwnerResource\Pages;
use App\Filament\Admin\Resources\BrikOwnerResource\RelationManagers;

class BrikOwnerResource extends Resource
{
    protected static ?string $model = BrikOwner::class;

    protected static ?string $navigationIcon = 'heroicon-o-rectangle-stack';

    protected static ?string $navigationGroup = 'BRIK OWNER';

    protected static ?string $modelLabel = 'Brik Owner';

    protected static ?string $pluralModelLabel = 'Brik Owners';

    protected static ?string $navigationLabel = 'Brik Owner';

    protected static ?int $navigationSort = 45;

    public static function form(Form $form): Form
    {
        return $form
            ->schema([
                Section::make()->schema([

                   Select::make('owner_id')
                    ->live()
                    ->label('Owner')
					->options(Client::where('is_owner', 1)->pluck('print_name', 'id'))
                    ->searchable()
					->required(),

                   Select::make('brik_unit_id')
                    ->live()
                    //->relationship('brik_unit', 'unit_name')
					->options(SelectDymanicNames::brikUnit())
                    ->afterStateUpdated(function(Set $set, $state, Get $get){
                        //if (is_null($get('id'))) {
                            self::fillSpvInfo($set, $state);
                            self::calculateCoinsForOwner($get, $set);
                        //}
                    })
                    ->afterStateHydrated(function(Set $set, $state, Get $get){
                        if (!is_null($get('id'))) {
                            self::fillSpvInfo($set, $state);
                            self::calculateCoinsForOwner($get, $set);
                        }
                    })
                    ->searchable()
                    ->required(),

                   TextInput::make('spv')
                    ->readonly(),

                   DatePicker::make('spv_formation_date')
                    ->readonly(),


                  TextInput::make('ownership')
                  ->label('Owner Ownership fraction')
                  ->numeric()
                  ->helperText('For Example fill only 1 if 1/11.')
                  ->required(),

                  TextInput::make('ownership_outof')
                  ->label('Ownership fraction total')
                  ->numeric()
                  ->default(11)
                  ->helperText('For Example fill only 11 if 1/11.')
                  ->required(),


                   TextInput::make('invested_amount')
                    ->numeric()
                    ->inputMode('decimal')
                    ->helperText('Enter amount in ₹.')
                    ->required(),


                    DatePicker::make('brik_buy_date')
                    ->live()
                    ->afterStateUpdated(function(Set $set, $state, Get $get){
                        //if (is_null($get('id'))) {
                            self::calculateCoinsForOwner($get, $set);
                        //}
                    })
                    ->afterStateHydrated(function(Set $set, $state, Get $get){
                        if (!is_null($get('id'))) {
                            self::calculateCoinsForOwner($get, $set);
                        }
                    }),

                    Select::make('referrer_id')
                    ->label('Referrer (If Any)')
                    ->options(function(Get $get){
                        return Client::where('is_owner', 1)->where('id','!=', $get('owner_id'))->pluck('full_name', 'id');
                    }),

                    // Select::make('relationship_manager_id')
                    // ->relationship('relationship_manager', 'name')
                    // ->required(),

                    TextInput::make('relationship_manager')
                    ->required(),

                    MarkdownEditor::make('remark')
                    ->columnSpan(2),

                    // TextInput::make('owner_coins_credited')
                    // ->numeric()
                    // ->helperText('Coins Calculated for owner'),

					Hidden::make('owner_coins_credited'),

                    Toggle::make('disabled')
                    ->label('Disable')
                    ->columnSpan(2),



                ])->columns(2),
                ]);
    }

    public static function table(Table $table): Table
    {
        return $table
            ->query(self::getModel()::query()->orderByDesc('updated_at'))
            ->columns([

                TextColumn::make('ownership_number')
                ->sortable()
                ->searchable(),

                TextColumn::make('owner.print_name')
                ->label('Owner')
                ->sortable()
                ->searchable(),

                TextColumn::make('owner.email')
                ->label('Owner Email')
                ->sortable()
                ->searchable(),

                // TextColumn::make('brik_unit.unit_name')
                // ->label('Brik Unit Name')
                // ->sortable()
                // ->searchable(),

				TextColumn::make('brik_unit.project.project_name')
                ->label('Project Name')
                ->sortable()
                ->searchable(),

                TextColumn::make('brik_unit.unit_number')
                ->label('Brik Unit Number')
                ->sortable()
                ->searchable(),

                TextColumn::make('invested_amount')
                ->sortable()
                ->searchable(),

                TextColumn::make('ownership_string')
                ->label('Ownership fraction')
                ->sortable()
                ->searchable(),
            ])
            ->filters([
                //
            ])
            ->actions([
                Tables\Actions\ActionGroup::make([
                    Tables\Actions\ViewAction::make(),
                    Tables\Actions\EditAction::make(),
                    Tables\Actions\DeleteAction::make()
                    ->action(function ($record) {

                        if ($record->childRecordCounts() > 0 ) {
                            Notification::make()
                                ->danger()
                                ->title('Brik Owner is in use')
                                ->body('Cannot delete this record because related records exist.')
                                ->send();

                                return;
                        }

                        Notification::make()
                                ->danger()
                                ->title('Deleted')
                                ->body('Deleted Brik Owner record.')
                                ->send();

                        $record->delete();
                    }),
                ])
            ])
            ->bulkActions([
                Tables\Actions\BulkActionGroup::make([
                    Tables\Actions\DeleteBulkAction::make(),
                ]),
            ])
            ->paginated([10, 25, 50, 100]);
    }

    public static function getRelations(): array
    {
        return [
            //
        ];
    }

    public static function getPages(): array
    {
        return [
            'index' => Pages\ListBrikOwners::route('/'),
            'create' => Pages\CreateBrikOwner::route('/create'),
            'edit' => Pages\EditBrikOwner::route('/{record}/edit'),
        ];
    }

    public static function fillSpvInfo(&$set, &$state)
    {
        $brik_unit = BrikUnit::with(['spv'])->find($state);
        if($brik_unit){
            if($brik_unit->spv){
                $set('spv', $brik_unit->spv->spv_name);
                $set('spv_formation_date', $brik_unit->spv->spv_formation_date);
            }
        }

    }

    public static function getAutoOwnerNumber()
    {
        $maxsq = BrikOwner::max('ownership_sq');
        $new_sq = $maxsq + 1;

        $prefix = 'BRK';
        $new_owner_number = $prefix.''.str_pad($new_sq, 5, '0', STR_PAD_LEFT);

        return $new_owner_number;

    }

    public static function calculateCoinsForOwner(&$get, &$set)
    {

        if(is_null($get('brik_unit_id')) || is_null($get('brik_buy_date'))){
            $set('owner_coins_credited', 0);
            //dump('1');
            return;
        }

        $unit = BrikUnit::find($get('brik_unit_id'));
        if(!$unit){
            $set('owner_coins_credited', 0);
            //dump('2');
            return;
        }

        if(!$unit->invest_active){
            $set('owner_coins_credited', 0);
            //dump('3');
            return;
        }

        $total_coins = $unit->total_coins_per_brik;
        $buy_date = $get('brik_buy_date');

        $permonthcoins = $total_coins / 12;

        $buy_datets = strtotime($buy_date);

        $buy_first_date = date("Y-m-01", strtotime($buy_date));

        $buy_first_datets = strtotime($buy_first_date);

        $buy_month = date('m', $buy_datets);

        if($buy_month <= 3)
		{
			$new_date = date("Y-03-31", $buy_datets);
            $new_datets = strtotime($new_date);
		}
		elseif($buy_month > 3)
		{
			$new_date = date("Y-03-31", strtotime("+1 year", $buy_datets));
            $new_datets = strtotime($new_date);
		}

        $diffdays = round(($new_datets - $buy_first_datets) / (60 * 60 * 24));

        $month = round($diffdays/30);


        $date = date('d', $buy_datets);


        if((int)$date > 15){
            $month = $month - 0.5;
        }

        $coins = round($permonthcoins * $month);

        $set('owner_coins_credited', $coins);
        //dd('4');
    }
}
