<?php

namespace App\Filament\Admin\Resources;

use Closure;
use \Carbon\Carbon;
use Filament\Forms;
use Filament\Tables;
use App\Actions\RU\RU;
use App\Models\Client;
use Filament\Forms\Get;
use Filament\Forms\Set;
use App\Models\BrikUnit;
use Filament\Forms\Form;
use Filament\Tables\Table;
use App\Models\BookingCoupon;
use App\Models\BookingBooking;
use Filament\Resources\Resource;
use Illuminate\Support\Facades\DB;
use Filament\Tables\Filters\Filter;
use App\Models\BookingBookingDetail;
use App\Models\BookingBookingRequest;
use App\Models\CoinWalletTransaction;
use Filament\Forms\Components\Select;
use Filament\Forms\Components\Toggle;
use Filament\Forms\Components\Section;
use Filament\Tables\Filters\Indicator;
use Filament\Forms\Components\Repeater;
use Filament\Tables\Columns\IconColumn;
use Filament\Tables\Columns\TextColumn;
use Filament\Forms\Components\TextInput;
use Filament\Notifications\Notification;
use Filament\Forms\Components\DatePicker;
use Filament\Tables\Filters\SelectFilter;
use Illuminate\Database\Eloquent\Builder;
use Filament\Tables\Filters\TernaryFilter;
use App\Actions\Utility\BookingDynamicPrice;
use Illuminate\Database\Eloquent\SoftDeletingScope;
use App\Filament\Admin\Resources\BookingBookingResource\Pages;
use App\Filament\Admin\Resources\BookingBookingResource\RelationManagers;

class BookingBookingResource extends Resource
{
    protected static ?string $model = BookingBooking::class;

    protected static ?string $navigationIcon = 'heroicon-o-rectangle-stack';

    protected static ?string $navigationGroup = 'BOOKING';

    protected static ?string $modelLabel = 'Booking';

    protected static ?string $pluralModelLabel = 'Bookings';

    protected static ?string $navigationLabel = 'Booking';

    protected static ?int $navigationSort = 61;

    public static function form(Form $form): Form
    {
        return $form
        ->schema([

            Section::make()->schema([
                Toggle::make('is_complimentary')
                ->live()
                ->disabledOn('edit')
                ->afterStateUpdated(function(Get $get, $state, Set $set){
                    $set('brik_unit_id', null);
                    $set('client_id', null);
                    $set('pay_by', null);
                    $set('date_check_in', null);
                    $set('date_check_out', null);
                    $set('price_per_night', null);
                    $set('booking_night_counts', null);
                    $set('booking_cost_in_rupee', null);
                    $set('coupon_amount', null);
                    $set('payable_amount', null);
                    $set('gst_percent', null);
                    $set('gst_amount', null);
                    $set('net_payable_amount', null);
                    $set('brik_coupon_id', null);
                    $set('paid_amount', null);
                    $set('balance_amount', null);
                })
                ->afterStateHydrated(function(Get $get, $state, Set $set){
                    if($get('booking_request_id')){
                        $book_req = BookingBookingRequest::find($get('booking_request_id'));
                        if($book_req){
                            if($book_req->is_complimentary){
                                $set('is_complimentary', true);
                            }else{
                                $set('is_complimentary', false);
                            }
                        }
                    }

                }),
            ]),

            Section::make()->schema([

                Select::make('brik_unit_id')
                ->label('Brik Unit')
                ->live()
                ->options(
                    BrikUnit::where('rent_active', 1)->pluck('booking_portal_display_name', 'id')
                )->afterStateUpdated(function(Get $get, $state, Set $set){

                    self::getGstPercent($get,$set);
                    self::singleCalculator($get,$set);



                })
                ->searchable()
                ->required()
                ->disabledOn('edit'),

                Select::make('client_id')
                ->live()
                ->label('User')
                ->options(
                    Client::query()->whereNotNull('first_name')->orderBy('first_name')->pluck('full_name', 'id')
                )
                ->afterStateUpdated(function(Get $get, $state, Set $set){

                    self::singleCalculator($get,$set);

                 })
                ->searchable()
                ->required(),

                Select::make('pay_by')
                ->options(function(Get $get, $state, Set $set){
                    if($get('is_complimentary')){
                        return [
                                  "Amount" => "Rupee",
                               ];
                    }else{
                        return [
                                  "Coin" => "Coin",
                                  "Amount" => "Rupee",
                               ];
                    }
                })
                ->live()
                ->afterStateUpdated(function(Get $get, $state, Set $set){

                    self::getGstPercent($get,$set);
                    self::singleCalculator($get,$set);

                 })
                ->required()
                ->disabledOn('edit'),

            ])->columnSpan(2)->columns(3),

            // Old Before Implement Update
            Section::make()->schema([

                DatePicker::make('date_check_in')
                ->live()
                ->native(false)
                ->afterStateUpdated(function(Get $get, $state, Set $set){
                   $set('date_check_out', '');

                   self::singleCalculator($get,$set);
                })
                ->minDate(function (Get $get, $state, Set $set){
                    if($get('id')){

                    }else{
                       return  date("Y-m-d");
                    }

                })
                ->disabledDates(function(Get $get){

                    if($get('brik_unit_id')){
                        return self::bookedDates($get('brik_unit_id'));
                    }
                    return [];
                })
                ->required(),

                DatePicker::make('date_check_out')
                ->live()
                ->native(false)
                ->disabledDates(function(Get $get){

                    if($get('brik_unit_id')){
                        return self::bookedDates($get('brik_unit_id'));
                    }
                    return [];
                })
                ->rule(



                fn (Get $get): Closure => function (string $attribute, $value, Closure $fail) use ($get){
                    $startDate = $get('date_check_in');
                    $endDate = $get('date_check_out');
                    $brik_unit_id = $get('brik_unit_id');

                    if (!$startDate || !$endDate) {
                        return null;
                    }

                    $conflict = BookingBookingRequest::where('brik_unit_id', $brik_unit_id)
                    ->where('booking_status', 'Active')
                    ->where(function ($query) use ($startDate, $endDate) {
                        $query->whereBetween('date_blocked_from', [$startDate, $endDate])
                            ->orWhereBetween('date_blocked_to', [$startDate, $endDate])
                            ->orWhere(function ($q) use ($startDate, $endDate) {
                                $q->where('date_blocked_from', '<=', $startDate)
                                  ->where('date_blocked_to', '>=', $endDate);
                            });
                    });

                    if($get('id')){
                        $conflict =  $conflict->where('id','!=', $get('booking_request_id'));
                    }

                    $conflict = $conflict->exists();

                    if($conflict){
                        $fail('The selected date range is already booked.');
                    }else{
                        if($brik_unit_id){

                           $brik_unit = BrikUnit::find($brik_unit_id);
                           if($brik_unit){
                                $ru_property_id = $brik_unit->ru_property_id;
                                if($ru_property_id){
                                    $ru = new RU;
                                    if($ru->isBookingDatesBlocked($ru_property_id, $startDate, $endDate)){
                                        $fail('The selected date range is already booked.');
                                    }
                                }
                           }

                        }
                    }
                }

                )
                ->afterStateUpdated(function(Get $get, $state, Set $set){
                //    if($get('date_check_in') && $get('date_check_out')){
                //         $price = self::calPricePerNight($get('brik_unit_id'), $get('date_check_in'), $get('date_check_out'));
                //         $set('price_per_night', $price);
                //     }
                //    self::calculateDays($get,$set);
                //    self::calculatePayables($get,$set);
                //    self::calculateCoinPayables($get,$set);
                    self::singleCalculator($get,$set);
                })
                ->minDate(function(Get $get, Set $set){
                    if($get('date_check_in')){
                       $newdate = Carbon::parse($get('date_check_in'))->addDays(1)->format('Y-m-d');
                       return $newdate;
                    }
                    return Carbon::today()->format('Y-m-d');
                })
                ->required(),

            ])->columnSpan(2)->columns(2),



            Section::make()->schema([

                TextInput::make('user_available_coins')
                ->live()
                ->gte('booking_cost_in_coin')
                ->numeric()
                ->afterStateHydrated(function(Get $get, $state, Set $set){

                    if(!is_null($get('id'))){

                        if(!is_null($get('client_id'))){

                            $client = Client::find($get('client_id'));

                            if($client){
                            $user_coins =  (int)$client->coins + (int)$get('booking_cost_in_coin');
                            $set('user_available_coins', $user_coins);
                            }
                        }
                    }
                 })
                 ->readOnly(),

                TextInput::make('coins_per_night')
                ->numeric(),

                TextInput::make('booking_night_counts')
                ->default(0)
                ->numeric(),

                TextInput::make('booking_cost_in_coin')
                ->default(0)
                ->numeric(),

            ])->columnSpan(1)->columns(1)->visible(function(Get $get){
                    if($get('pay_by') == 'Coin'){
                        return true;
                    }else{
                        return false;
                    }
            }),

            Section::make()->schema([

                TextInput::make('price_per_night')
                ->label('Per Night Cost (₹)')
                ->live(onBlur: true)
                ->afterStateUpdated(function(Get $get, $state, Set $set){
                    //self::calculatePayables($get,$set);
                    self::singleCalculator($get,$set);
                 })
                ->numeric(),

                TextInput::make('booking_night_counts')
                ->live(onBlur: true)
                ->afterStateUpdated(function(Get $get, $state, Set $set){
                    //self::calculatePayables($get,$set);
                    self::singleCalculator($get,$set);
                 })
                ->label('Number of Booking Nights')
                ->numeric()
                ->readOnly(),

                TextInput::make('booking_cost_in_rupee')
                ->label('Booking Amount (₹)')
                ->numeric()
                ->readOnly(),

                TextInput::make('coupon_amount')
                ->live(onBlur: true)
                ->label('Discount (₹)')
                ->numeric()
                 ->afterStateUpdated(function(Get $get, $state, Set $set){
                    //self::calculatePayables($get,$set);
                    self::singleCalculator($get,$set);
                 }),


                TextInput::make('payable_amount')
                ->label('Payable Amount (Booking Amount (₹) - Discount (₹))')
                ->numeric(),

                TextInput::make('gst_percent')
                ->live(onBlur: true)
                ->label('GST %')
                ->numeric()
                ->afterStateUpdated(function(Get $get, $state, Set $set){
                    //self::calculatePayables($get,$set);
                    self::singleCalculator($get,$set);
                 }),

                TextInput::make('gst_amount')
                ->numeric(),

                TextInput::make('net_payable_amount')
                ->numeric(),

                TextInput::make('paid_amount')
                ->live(onBlur: true)
                ->numeric()
                ->maxValue(function(Get $get){
                    return $get('net_payable_amount');
                })
                ->afterStateUpdated(function(Get $get, $state, Set $set){
                   $balance =  ((float)$get('net_payable_amount') - (float)$get('paid_amount'));
                   $set('balance_amount', $balance);

                 })
                ->required(),

                TextInput::make('balance_amount')
                ->numeric()
                ->readOnly(),

            ])->columnSpan(1)->columns(1)->visible(function(Get $get){
                if($get('pay_by') == 'Amount'){
                    return true;
                }else{
                    return false;
                }
            }),

            Section::make()->schema([

                Select::make('brik_coupon_id')
                ->label('Discount Coupon')
                ->live()
                ->options(function(Get $get, Set $set){

                    if($get('date_check_in') && $get('date_check_out')){
                        return BookingCoupon::where('valid_from','<=', date('Y-m-d'))->where('valid_to','>=', date('Y-m-d'))
                        ->where('booking_valid_from','<=', $get('date_check_in'))->where('booking_valid_to','>=', $get('date_check_out'))->where('active', 1)->pluck('select_name', 'id');
                    }

                })
                ->afterStateUpdated(function(Get $get, $state, Set $set){
                    //self::calculatePayables($get,$set);
                    self::singleCalculator($get,$set);
                 })->visible(function(Get $get){
                    if($get('pay_by') == 'Amount'){
                        return true;
                    }else{
                        return false;
                    }
                })
                ,

                TextInput::make('adult')
                ->default(1)
                ->numeric(),

                TextInput::make('children')
                ->default(0)
                ->numeric(),

                TextInput::make('infants')
                ->default(0)
                ->numeric(),

                TextInput::make('pets')
                ->default(0)
                ->numeric(),


            ])->columnSpan(1)->visible(function(Get $get){
                if($get('pay_by') == 'Amount' || $get('pay_by') == 'Coin'){
                    return true;
                }else{
                    return false;
                }
            }),





        ])->columns(2);
    }

    public static function table(Table $table): Table
    {
        return $table
            ->query(self::getModel()::query()->orderByDesc('updated_at'))
            ->columns([

                TextColumn::make('booking_status')
                ->sortable()
                ->searchable(),

                TextColumn::make('brik_unit.booking_portal_display_name')
                ->sortable()
                ->searchable(),

                TextColumn::make('date_check_in')
                ->date()
                ->sortable()
                ->searchable(),

                TextColumn::make('date_check_out')
                ->date()
                ->sortable()
                ->searchable(),

                TextColumn::make('pay_by')
                ->formatStateUsing(function($state){
                    if($state == 'Amount')
                    {
                        return 'Rupee';
                    }else{
                        return $state;
                    }
                })
                ->sortable()
                ->searchable(),

                IconColumn::make('booking_request.is_complimentary')
                ->label('Is Complimentary')
                ->boolean()
                ->sortable(),

                TextColumn::make('client.full_name')
                ->label('Guest')
                ->sortable()
                ->searchable(),

            ])
            ->filters([
                SelectFilter::make('brik_unit_id')
                ->label('Brik Unit')
                ->options(
                    BrikUnit::where('invest_active', 1)->where('rent_active', 1)
                    ->pluck('booking_portal_display_name', 'id'))
                ->multiple(),

                Filter::make('Booking Date')
                ->form([
                    DatePicker::make('booking_from'),
                    DatePicker::make('booking_until'),
                ])
                ->query(function (Builder $query, array $data): Builder {
                    return $query
                        ->when(
                            $data['booking_from'],
                            fn (Builder $query, $date): Builder => $query->whereDate('date_check_in', '>=', $date),
                        )
                        ->when(
                            $data['booking_until'],
                            fn (Builder $query, $date): Builder => $query->whereDate('date_check_out', '<=', $date),
                        );
                })
                ->indicateUsing(function (array $data): array {
                    $indicators = [];

                    if ($data['booking_from'] ?? null) {
                        $indicators[] = Indicator::make('Booking from ' . Carbon::parse($data['booking_from'])->toFormattedDateString())
                            ->removeField('booking_from');
                    }

                    if ($data['booking_until'] ?? null) {
                        $indicators[] = Indicator::make('Booking until ' . Carbon::parse($data['booking_until'])->toFormattedDateString())
                            ->removeField('booking_until');
                    }

                    return $indicators;
                }),

                SelectFilter::make('booking_status')
                ->options([
                    'Active' => 'Active',
                    'Cancel' => 'Cancel',
                    'Initiated' => 'Initiated',
                ]),

                // Filter::make('complementory')
                // ->label('Only Complementory Bookings')
                // ->query(function (Builder $query, array $data): Builder {
                //     if (isset($data['complementory']) && $data['complementory'] === true) {
                //         return $query->whereHas('booking_request', function (Builder $query) {
                //             $query->where('is_complimentary', 1);
                //         });
                //     }
                //     return $query;
                // })
                // ->indicator('Only Complementory Bookings')
                // ->default(false),

            ])
            ->actions([
                Tables\Actions\ActionGroup::make([
                    Tables\Actions\ViewAction::make(),
                    Tables\Actions\EditAction::make(),
                    Tables\Actions\DeleteAction::make()
                     ->action(function ($record) {

                            if($record->pay_by == 'Coin')
                            {
                                CoinWalletTransaction::where('booking_id', $record->id)->delete();

                                $client = Client::find($record->client_id);
                                $client->coins = ($client->coins + $record->booking_cost_in_coin);
                                $client->save();
                            }

                            $delbookreq = BookingBookingRequest::find($record->booking_request_id);
                            $delbookreq->delete();

                            BookingBookingDetail::where('booking_id', $record->id)->delete();

                            Notification::make()
                                    ->danger()
                                    ->title('Deleted')
                                    ->body('Booking Deleted.')
                                    ->send();

                            $record->delete();


                        }),
                ])
            ])
            ->bulkActions([
                Tables\Actions\BulkActionGroup::make([
                    Tables\Actions\DeleteBulkAction::make(),
                ]),
            ]);
    }

    public static function getRelations(): array
    {
        return [
            //
        ];
    }

    public static function getPages(): array
    {
        return [
            'index' => Pages\ListBookingBookings::route('/'),
            'create' => Pages\CreateBookingBooking::route('/create'),
            'edit' => Pages\EditBookingBooking::route('/{record}/edit'),
        ];
    }

    public static function calculateDays(&$get,&$set)
    {
        $checkin_date = $get('date_check_in');
        $checkout_date = $get('date_check_out');

        if($checkin_date && $checkout_date){
            $from = strtotime($checkin_date);
            $to = strtotime($checkout_date);
            $diff = round(($to - $from) / (60 * 60 * 24));

            if($from === $to){
                $set('booking_night_counts', 1);
            }else{
                $set('booking_night_counts', $diff);
            }


        }
    }

    public static function calculatePayables(&$get,&$set)
    {
        $price_per_night = $get('price_per_night');
        $booking_night_counts = $get('booking_night_counts');
        $discount_coupon_id = $get('brik_coupon_id');
        $gst_percent = $get('gst_percent');
        $paid_amount = $get('paid_amount');
        $balance_amount = 0;




        if($price_per_night && $booking_night_counts){

            $booking_cost = $price_per_night * $booking_night_counts;
            $set('booking_cost_in_rupee', $booking_cost);
            $coupon_amount = 0;
            if($discount_coupon_id){
                $coupon = BookingCoupon::find($discount_coupon_id);
                if($coupon){
                    if($coupon->discount_mode == 'Rupee'){
                        $coupon_amount = $coupon->value;
                    }else if($coupon->discount_mode == 'Percent'){
                        $coupon_amount = (($coupon->value * $booking_cost) / 100);
                    }
                }
            }else{
                $coupon_amount = $get('coupon_amount') ? $get('coupon_amount') : 0;
            }

                $payable_amount = $booking_cost - $coupon_amount;
                $gst_amount = ($payable_amount * ($gst_percent / 100));
                $net_payable = $payable_amount + $gst_amount;

                if(!is_null($paid_amount)){
                    if((float)$paid_amount > 0){
                        $balance_amount = $net_payable - (float)$paid_amount;
                    }
                }


                $set('coupon_amount', $coupon_amount);
                $set('payable_amount', $payable_amount);
                $set('gst_amount', $gst_amount);
                $set('net_payable_amount', $net_payable);
                $set('balance_amount', $balance_amount);

        }else{
            $set('booking_cost_in_rupee', 0);
            $set('coupon_amount', 0);
            $set('payable_amount', 0);
            $set('gst_amount', 0);
            $set('net_payable_amount', 0);
            $set('balance_amount', 0);
        }
    }

    public static function calculateCoinPayables(&$get,&$set)
    {
        $coins_per_night = $get('coins_per_night') ? $get('coins_per_night') : 0;
        $booking_night_counts = $get('booking_night_counts') ? $get('booking_night_counts') : 0;
        $booking_cost_in_coin = $coins_per_night * $booking_night_counts;

        $set('booking_cost_in_coin', $booking_cost_in_coin);


    }

    public static function calPricePerNight($brik_unit_id, $date_check_in, $date_check_out){

        // if($brik_unit_id && $date_check_in && $date_check_out){

        //     $price = new BookingDynamicPrice($brik_unit_id, $date_check_in, $date_check_out);
        //     $totalPrice = $price->extractDates()->groupDatesWithPrice()->finalGrouping()->addDaysAndPrice()->sum();
        //     $nights = self::countDays($date_check_in,$date_check_out);
        //     $price_per_night = ($totalPrice / $nights);

        //     return $price_per_night;

        // }

        return 0;
    }

    public static function  complimentoryPricePerNight(&$get,&$set){

        $complimentary_per_night_inr = 0;
        $unit = BrikUnit::with(['brik','brik.propertyType'])->find($get('brik_unit_id'));
        if($unit){
            if($unit->brik){
                if($unit->brik->propertyType){
                    $complimentary_per_night_inr = $unit->brik->propertyType->complimentary_per_night_inr;
                }
            }
        }
        $set('price_per_night', $complimentary_per_night_inr);
    }

    // Before Update Implement
    // public static function bookedDates($unit_id)
    // {
    //     $date = date('Y-m-d');
    //     $q = "WITH RECURSIVE
    //             cte AS ( SELECT `date_blocked_from`, `date_blocked_to`, `date_blocked_from` single_date
    //                     FROM booking_booking_requests WHERE booking_status = 'Active' AND `brik_unit_id` = {$unit_id} AND `date_blocked_to` > {$date}
    //                     UNION ALL
    //                     SELECT `date_blocked_from`, `date_blocked_to`, single_date + INTERVAL 1 DAY
    //                     FROM cte
    //                     WHERE single_date < `date_blocked_to` )

    //             SELECT DISTINCT single_date
    //             FROM cte
    //             ORDER BY 1;";

    //     $res = collect(DB::select($q))->pluck('single_date')->toArray();

    //     return $res;
    // }

    public static function bookedDates($unit_id, $excludeId = null)
    {
        $today = date('Y-m-d');

        $excludeCondition = '';
        if ($excludeId) {
            $excludeCondition = "AND id != {$excludeId}";
        }

        $q = "
            WITH RECURSIVE cte AS (
                SELECT date_blocked_from, date_blocked_to, date_blocked_from AS single_date
                FROM booking_booking_requests
                WHERE booking_status = 'Active'
                AND brik_unit_id = {$unit_id}
                AND date_blocked_to >= '{$today}'
                {$excludeCondition}

                UNION ALL

                SELECT cte.date_blocked_from, cte.date_blocked_to, DATE_ADD(cte.single_date, INTERVAL 1 DAY)
                FROM cte
                WHERE cte.single_date < cte.date_blocked_to
            )
            SELECT DISTINCT single_date
            FROM cte
            ORDER BY single_date;
        ";

        return collect(DB::select($q))->pluck('single_date')->toArray();
    }

    public static function countDays($fromdate,$todate)
    {
        $from = strtotime($fromdate);
        $to = strtotime($todate);
        $diff = round(($to - $from) / (60 * 60 * 24));

        if($from === $to){
            return 1;
        }else{
            return (int)$diff;
        }
    }

    public static function singleCalculator(&$get,&$set)
    {
        if($get('pay_by') == 'Coin'){

            if(is_null($get('id'))){

                if($get('client_id')){

                    $client = Client::find($get('client_id'));

                    if($client){
                        $set('user_available_coins', $client->coins);
                    }
                }
            }
        }

        if($get('brik_unit_id')){

            $unit = BrikUnit::find($get('brik_unit_id'));

            if($unit){
                $set('coins_per_night', $unit->per_night_coin);
            }
        }

        if($get('pay_by') == 'Coin'){
            self::calculateCoinPayables($get,$set);
        }

        if($get('is_complimentary')){

            self::complimentoryPricePerNight($get,$set);

        }else{

            if($get('date_check_in') && $get('date_check_out')){

                $price = self::calPricePerNight($get('brik_unit_id'), $get('date_check_in'), $get('date_check_out'));

                if(!$get('price_per_night')){
                    $set('price_per_night', $price);
                }


            }
        }


        self::calculateDays($get,$set);
        self::calculatePayables($get,$set);
        self::calculateCoinPayables($get,$set);
    }

    public static function getGstPercent(&$get,&$set)
    {
        //dd($get('pay_by'));
        if($get('pay_by') == 'Amount'){

            if($get('brik_unit_id')){

                $unit = BrikUnit::find($get('brik_unit_id'));

                if($unit){
                    $set('gst_percent', $unit->gst_percent);
                }

            }
        }


        self::calculateDays($get,$set);
        self::calculatePayables($get,$set);
        self::calculateCoinPayables($get,$set);
    }
}
