<?php

namespace App\Filament\Admin\Resources;

use Filament\Forms;
use Filament\Tables;
use App\Models\Client;
use Filament\Forms\Get;
use Filament\Forms\Set;
use Filament\Forms\Form;
use App\Models\BrikOwner;
use Filament\Tables\Table;
use App\Models\BrikInstallment;
use Filament\Resources\Resource;
use App\Models\BrikPaymentmyplan;
use Filament\Forms\Components\Select;
use Filament\Forms\Components\Toggle;
use App\Models\BrikInstallmentPayment;
use Filament\Forms\Components\Section;
use Filament\Forms\Components\Repeater;
use Filament\Tables\Columns\IconColumn;
use Filament\Tables\Columns\TextColumn;
use Filament\Forms\Components\TextInput;
use Filament\Forms\Components\DatePicker;
use Illuminate\Database\Eloquent\Builder;
use App\Actions\Utility\SelectDymanicNames;
use Illuminate\Database\Eloquent\SoftDeletingScope;
use App\Filament\Admin\Resources\BrikInstallmentResource\Pages;
use App\Filament\Admin\Resources\BrikInstallmentResource\RelationManagers;

class BrikInstallmentResource extends Resource
{
    protected static ?string $model = BrikInstallment::class;

    protected static ?string $navigationIcon = 'heroicon-o-rectangle-stack';

    protected static ?string $navigationGroup = 'BRIK OWNER';

    protected static ?string $modelLabel = 'Payment Installment';

    protected static ?string $pluralModelLabel = 'Payment Installments';

    protected static ?string $navigationLabel = 'Payment Installment';

    protected static ?int $navigationSort = 47;

    public static function form(Form $form): Form
    {
        return $form
        ->schema([
            Section::make()->schema([

               Select::make('payment_plan_id')
                ->live()
                ->label('Brik Owner Payment Plan')
                ->options(function(){
                   //$paymentplans = BrikPaymentmyplan::query()->pluck('name', 'id');
                   $paymentplans = SelectDymanicNames::paymentPlan();
                   //dd($paymentplans);
                   return $paymentplans;
                })
                ->afterStateUpdated(function(Set $set, $state, Get $get){
                    if (is_null($get('id'))) {
                        self::fillInstallmentNo($set, $state);
                        self::fillWalletInfo($set, $state);
                    }
                })
                ->afterStateHydrated(function(Set $set, $state, Get $get){
                    self::fillWalletInfo($set, $state);
                })
                ->searchable()
                ->required()
                ->columnSpanFull(),

               TextInput::make('installment')
                ->label('Installment No.')
                ->numeric(),

               TextInput::make('installment_amount')
               ->live(onBlur: true)
               ->afterStateUpdated(function (Get $get, Set $set){
                    self::calculatePayable($get, $set);
                    self::calculateBalance($get, $set);
               })
               ->numeric()
               ->inputMode('decimal')
               ->required(),

               DatePicker::make('due_date'),

               TextInput::make('interest_on_installment')
               ->live(onBlur: true)
               ->afterStateUpdated(function (Get $get, Set $set){
                    self::calculatePayable($get, $set);
                    self::calculateBalance($get, $set);
               })
               ->numeric()
               ->inputMode('decimal'),

               Section::make()->schema([

                TextInput::make('installment_referal_discount')
                ->lte('wallet_current_balance')
                ->live(onBlur: true)
                ->numeric()
                ->inputMode('decimal')
                ->afterStateUpdated(function (Get $get, Set $set){
                    self::calculatePayable($get, $set);
                    self::calculateBalance($get, $set);
                }),

                TextInput::make('wallet_current_balance')
                ->readonly()

              ])->columns(2),

              Section::make()->schema([

                    TextInput::make('payable_amount')
                    ->numeric()
                    ->readOnly()
                    ->inputMode('decimal')
                    ->required(),

                    TextInput::make('amount_paid')
                    ->lte('payable_amount')
                    ->readOnly(),

                    TextInput::make('balance_amount')
                    ->afterStateHydrated(function (Get $get, Set $set) {
                        $set('balance_amount', ( $get('payable_amount') - $get('amount_paid')));
                    })
                    ->readOnly(),

               ])->columns(3),



            //    Toggle::make('active')
            //    ->live()
            //    ->afterStateUpdated(function(Set $set, $state, Get $get){
            //         if($state == false){
            //             $set('installment_paid', 0);
            //             $set('amount_paid', '');
            //             $set('paid_date', '');
            //         }
            //     })
            //    ->columnSpanFull(),




            ])->columns(2),

            Section::make()->schema([

                Repeater::make('payments')
                ->relationship('payments')
                ->schema([

                    TextInput::make('paid_amount')
                    ->numeric()
                    ->inputMode('decimal')
                    ->required(),

                    DatePicker::make('date')
                    ->required(),

                ])
                ->live(onBlur: true)
                ->afterStateUpdated(function (Get $get, Set $set, $state) {

                    $total = collect($state)->sum(fn ($item) => floatval($item['paid_amount'] ?? 0));

                    $balance = $get('payable_amount') - $total;

                    $set('amount_paid', $total);

                    $set('balance_amount', $balance);
                })
                ->columns(2),

            ])->columnSpan(2),

            // Section::make()->schema([
            //     Toggle::make('installment_paid')
            //    ->live()
            //    ->afterStateUpdated(function(Set $set, $state, Get $get){
            //         if($state == false){
            //             $set('amount_paid', '');
            //             $set('paid_date', '');
            //         }
            //     })
            //    ->columnSpanFull(),
            //    //->disabled(fn (Get $get): bool => !$get('active')),

            //    TextInput::make('amount_paid')
            //    ->numeric()
            //    ->inputMode('decimal')
            //    ->required(fn (Get $get): bool => $get('installment_paid')),
            //    //->disabled(fn (Get $get): bool => !$get('active')),

            //    DatePicker::make('paid_date')
            //    ->required(fn (Get $get): bool => $get('installment_paid')),
            //    //->disabled(fn (Get $get): bool => !$get('active')),

            // ])->columns(2),
        ]);
    }

    public static function table(Table $table): Table
    {
        return $table
            ->query(self::getModel()::query()->orderByDesc('updated_at'))
            ->columns([

                TextColumn::make('installment')
                ->label('Installment No.')
                ->searchable(),

                TextColumn::make('paymentPlan.brik_owner.owner.print_name')
                ->label('Brik Owner Name')
                ->searchable(),

                TextColumn::make('paymentPlan.brik_owner.owner.email')
                ->label('Brik Owner Email')
                ->searchable(),
				
				TextColumn::make('paymentPlan.brik_owner.brik_unit.spv.spv_name')
                ->label('SPV')
                ->searchable(),
				
				TextColumn::make('paymentPlan.brik_owner.brik_unit.booking_portal_display_name')
                ->label('Booking Portal Display Name')
                ->searchable(),



                // TextColumn::make('paymentPlan.name')
                // ->sortable()
                // ->searchable(),

                TextColumn::make('installment_amount')
                ->sortable()
                ->searchable(),

                TextColumn::make('due_date')
                ->date()
                ->searchable(),

                IconColumn::make('active')->boolean(),

                IconColumn::make('installment_paid')
                ->boolean()
                ->sortable(),



                TextColumn::make('interest_on_installment')
                ->sortable()
                ->searchable(),

            ])
            ->filters([
                //
            ])
            ->actions([
                Tables\Actions\ActionGroup::make([
                    Tables\Actions\ViewAction::make(),
                    Tables\Actions\EditAction::make(),
                    Tables\Actions\DeleteAction::make(),
					Tables\Actions\Action::make('receipt')
                    ->label('Receipt')
                    ->icon('heroicon-o-link')
                    ->url(fn ($record) => route('installment.receipt', $record->id))
                    ->openUrlInNewTab()
                    ->visible(fn ($record) => $record->installment_paid),
                ])
            ])
            ->bulkActions([
                Tables\Actions\BulkActionGroup::make([
                    Tables\Actions\DeleteBulkAction::make(),
                ]),
            ])
            ->paginated([10, 25, 50, 100])
			->persistSearchInSession();
    }

    public static function getRelations(): array
    {
        return [
            //
        ];
    }

    public static function getPages(): array
    {
        return [
            'index' => Pages\ListBrikInstallments::route('/'),
            'create' => Pages\CreateBrikInstallment::route('/create'),
            'edit' => Pages\EditBrikInstallment::route('/{record}/edit'),
        ];
    }

    public static function fillInstallmentNo(&$set, &$state)
    {
        $count = BrikInstallment::where('payment_plan_id',$state)->count() + 1;
        $set('installment', $count);
    }

    public static function fillWalletInfo(&$set, &$state)
    {
        $cash_wallet = 0;
        $paymentplan = BrikPaymentmyplan::find($state);

        if($paymentplan){
            $brik_owner = BrikOwner::find($paymentplan->brik_owner_id);

            if($brik_owner){

                $owner = Client::find($brik_owner->owner_id);

                if($owner){
                    $cash_wallet = $owner->wallet_amount;
                }
            }
        }

        $set('wallet_current_balance', $cash_wallet);
    }

    public static function calculatePayable(&$get, &$set)
    {
        $installment_amount = is_numeric($get('installment_amount')) ? $get('installment_amount') : 0;
        $interest_on_installment = is_numeric($get('interest_on_installment')) ? $get('interest_on_installment') : 0;
        $installment_referal_discount = is_numeric($get('installment_referal_discount')) ? $get('installment_referal_discount') : 0;

        $payable_amount =
        $installment_amount +
        $interest_on_installment -
        $installment_referal_discount;

        $set('payable_amount', $payable_amount);
    }

    public static function calculateBalance(&$get, &$set)
    {
        $installment_amount = is_numeric($get('installment_amount')) ? $get('installment_amount') : 0;
        $interest_on_installment = is_numeric($get('interest_on_installment')) ? $get('interest_on_installment') : 0;
        $installment_referal_discount = is_numeric($get('installment_referal_discount')) ? $get('installment_referal_discount') : 0;

        $payable_amount =
        $installment_amount +
        $interest_on_installment -
        $installment_referal_discount;

        if(!is_null($get('id'))){
            $payment_sum = BrikInstallmentPayment::where('brik_installment_id', $get('id'))->sum('paid_amount');
        }else{
            $payment_sum = 0;
        }

        $balance_amount = $payable_amount - $payment_sum;


        $set('balance_amount', $balance_amount);
    }
}
