<?php

namespace App\Http\Controllers\Api\Booking;

use App\Actions\RU\RU;
use App\Models\Client;
use App\Models\BrikSpv;
use App\Models\BrikUnit;
use App\Models\BrikProject;
use Illuminate\Http\Request;
use App\Models\BookingBooking;
use Illuminate\Support\Facades\DB;
use App\Http\Controllers\Controller;
use App\Models\BookingBookingDetail;
use Illuminate\Support\Facades\Auth;
use App\Models\BookingBookingRequest;
use App\Models\CoinWalletTransaction;
use App\Actions\Utility\FinancialYear;
use App\Actions\Payment\BookingPayment;
use Illuminate\Support\Facades\Redirect;
use Illuminate\Support\Facades\Validator;
use App\Actions\Utility\BookingDynamicPrice;
use App\Http\Resources\BookingHistoryResource;
use App\Http\Resources\BookingRequestResource;
use App\Models\BookingCancellationPolicyMaster;

class BookingController extends Controller
{
    public function bookedDates($unit_id)
    {
        $date = date('Y-m-d');
        $q = "WITH RECURSIVE
                cte AS ( SELECT `date_blocked_from`, `date_blocked_to`, `date_blocked_from` single_date 
                        FROM booking_booking_requests WHERE booking_status = 'Active' AND `brik_unit_id` = {$unit_id} AND `date_blocked_to` > {$date}
                        UNION ALL
                        SELECT `date_blocked_from`, `date_blocked_to`, single_date + INTERVAL 1 DAY
                        FROM cte
                        WHERE single_date < `date_blocked_to` )

                SELECT DISTINCT single_date
                FROM cte
                ORDER BY 1;";

        $res = collect(DB::select($q))->pluck('single_date')->toArray();
        
        return response()->json($this->throwSuccessResponse(["data" => ["blocked_dates" => $res]]));
    }

    

    public function preBooking(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'date_check_in' => 'required|date',
            'date_check_out' => 'required|date|after:date_check_in',
            'use_coin' => 'required',
            'brik_unit_id' => 'required',
        ]);

        if ($validator->fails()) {
            return response()->json($this->throwValidationFailedResponse($this->modifyValidationErrors($validator->messages()->get('*'))));
        }
       
       
        $unit = BrikUnit::find($request->brik_unit_id);
        $client = $request->user();
        
        if(!$client){
            return response()->json($this->throwFailedResponse("User information not found in server."));
        }

        if(!$unit){
            return response()->json($this->throwFailedResponse("Brik Unit not found in server."));
        }
        
        $booking_days = $this->countDays($request->date_check_in,$request->date_check_out);
        if($request->use_coin == 1){

            $days_client_book_using_coins = $this->countDaysCanBookUsingCoins($client, $unit);
            //dd($days_client_book_using_coins, $booking_days);
            
            if($days_client_book_using_coins >= $booking_days){
                // only for coins
                //dd('only coin');
                $coin_structure = $this->coinBookingStucture($request->date_check_in, $request->date_check_out, $booking_days, $unit->per_night_coin);
            }else if($days_client_book_using_coins == 0){

                $payment_structure = $this->paymentBookingStucture($request, $request->date_check_in, $request->date_check_out, $booking_days);

            }
            else{
              
                //dd('both');
                // both by payment & by coins
               $separated_dates = $this->separateCoinAndPaymentDates($request, $days_client_book_using_coins, $booking_days);

               $coin_structure = $this->coinBookingStucture($separated_dates["coin_checkin_date"], $separated_dates["coin_checkout_date"], $separated_dates["coin_pay_days"], $unit->per_night_coin);

               $payment_structure = $this->paymentBookingStucture($request, $separated_dates["payment_checkin_date"], $separated_dates["payment_checkout_date"], $separated_dates["payment_pay_days"]);

            }
        }else{
            //dd('only payment');
            // only for payment
            $payment_structure = $this->paymentBookingStucture($request, $request->date_check_in, $request->date_check_out, $booking_days);
        }

        $return_data = [
            "date_check_in" => $request->date_check_in,
            "date_check_out" => $request->date_check_out,
            "brik_unit_id" => $request->brik_unit_id,
            "adult" => $request->adult,
            "children" => $request->children,
            "infants" => $request->infants,
            "pets" => $request->pets,
            "payment_booking" => isset($payment_structure) ? $payment_structure : null,
            "coin_booking" => isset($coin_structure) ? $coin_structure : null,
        ];

        return response()->json($this->throwSuccessResponse(["data" => $return_data]));
    }

    public function bookingCostCal(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'brik_unit_id' => 'required',
            'date_check_in' => 'required',
            'date_check_out' => 'required',
        ]);

        if ($validator->fails()) {
            return response()->json($this->throwValidationFailedResponse($this->modifyValidationErrors($validator->messages()->get('*'))));
        }

        $price = new BookingDynamicPrice($request->brik_unit_id, $request->date_check_in, $request->date_check_out);
        $totalPrice = $price->extractDates()->groupDatesWithPrice()->finalGrouping()->addDaysAndPrice()->sum();
        $nights = $this->countDays($request->date_check_in,$request->date_check_out);
        $price_per_night = ($totalPrice / $nights);
       
        //->groupDatesWithPrice()->finalGrouping()->addDaysAndPrice();

        return response()->json($this->throwSuccessResponse(["data" => [
            'price_per_night' => $price_per_night,
        ]]));
    }

    public function booking(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'brik_unit_id' => 'required',
        ]);

        if ($validator->fails()) {
            return response()->json($this->throwValidationFailedResponse($this->modifyValidationErrors($validator->messages()->get('*'))));
        }

        $date_blocked_from = $request->date_check_in;
        $date_blocked_to = date("Y-m-d",strtotime("-1 day ".$request->date_check_out));

        $conflict = BookingBookingRequest::where('brik_unit_id', $request->brik_unit_id)->where('booking_status', 'Active')
                    ->where(function ($query) use ($date_blocked_from, $date_blocked_to) {
                        $query->whereBetween('date_blocked_from', [$date_blocked_from, $date_blocked_to])
                            ->orWhereBetween('date_blocked_to', [$date_blocked_from, $date_blocked_to])
                            ->orWhere(function ($q) use ($date_blocked_from, $date_blocked_to) {
                                $q->where('date_blocked_from', '<=', $date_blocked_from)
                                  ->where('date_blocked_to', '>=', $date_blocked_to);
                            });
                    })->exists();

        if($conflict){
         
            return response()->json($this->throwFailedResponse('The selected date range is already booked.'));
        }

        // RU check booking availibility
        $ru_sync_required = 0;
        $brikunit = BrikUnit::find($request->brik_unit_id);
        if($brikunit){

            if($brikunit->ru_property_id){

                $ru_sync_required = 1;
                $ru = new RU;
                if($ru->isBookingDatesBlocked($brikunit->ru_property_id, $request->date_check_in, $request->date_check_out)){
                    return response()->json($this->throwFailedResponse('The selected date range is already occupied.'));
                }

            }
                
        }
        
       
        
        if(is_null($request->coin_booking) && is_null($request->payment_booking)){
            return response()->json($this->throwFailedResponse('Coin Booking & Payment booking has no data.'));
        }

        DB::beginTransaction();

        try
        {
            if(!is_null($request->coin_booking) && is_null($request->payment_booking)){
                $coinBookingStatus = 'Active';
            }else{
                $coinBookingStatus = 'Initiated';
            }

            $booking_request = new BookingBookingRequest;
            $booking_request->brik_unit_id = $request->brik_unit_id;
            $booking_request->booking_from_date = $request->date_check_in;
            $booking_request->booking_to_date = $request->date_check_out;
            $booking_request->date_blocked_from = $date_blocked_from;
            $booking_request->date_blocked_to = $date_blocked_to;
            $booking_request->booking_status = $coinBookingStatus;
            $booking_request->ru_sync_required = $ru_sync_required;
            $booking_request->save();

            $has_payment_booking = false;
            if($request->coin_booking){

                $coinBooking = new BookingBooking;
                $coinBooking->booking_request_id = $booking_request->id;
                $coinBooking->date_check_in = $request->coin_booking["date_check_in"];
                $coinBooking->date_check_out = $request->coin_booking["date_check_out"];
                $coinBooking->coins_per_night = $request->coin_booking["coins_per_night"];
                $coinBooking->booking_night_counts = $request->coin_booking["booking_night_counts"];
                $coinBooking->booking_cost_in_coin = $request->coin_booking["booking_cost_in_coin"];
                $coinBooking->pay_by = 'Coin';
                $coinBooking->adult = $request->adult ? $request->adult : 0;
                $coinBooking->children = $request->children ? $request->children : 0;
                $coinBooking->infants = $request->infants ? $request->infants : 0;
                $coinBooking->pets = $request->pets ? $request->pets : 0;
                $coinBooking->booking_status = $coinBookingStatus;
                $coinBooking->brik_unit_id = $request->brik_unit_id;
                $coinBooking->client_id = $request->user()->id;
                $coinBooking->save();

            }

            if($request->payment_booking){

                $payBooking = new BookingBooking;
                $payBooking->booking_request_id = $booking_request->id;
                $payBooking->date_check_in = $request->payment_booking["date_check_in"];
                $payBooking->date_check_out = $request->payment_booking["date_check_out"];
                //$payBooking->price_per_night = $request->payment_booking["price_per_night"];
                $payBooking->booking_night_counts = $request->payment_booking["booking_night_counts"];
                $payBooking->booking_cost_in_rupee = $request->payment_booking["booking_cost_in_rupee"];
                $payBooking->brik_coupon_id = $request->payment_booking["brik_coupon_id"];
                $payBooking->coupon_amount = $request->payment_booking["coupon_amount"];
                $payBooking->payable_amount = $request->payment_booking["payable_amount"];
                $payBooking->gst_amount = $request->payment_booking["gst_amount"];
                $payBooking->net_payable_amount = $request->payment_booking["net_payable_amount"];
                $payBooking->pay_by = 'Amount';
                $payBooking->adult = $request->adult ? $request->adult : 0;
                $payBooking->children = $request->children ? $request->children : 0;
                $payBooking->infants = $request->infants ? $request->infants : 0;
                $payBooking->pets = $request->pets ? $request->pets : 0;
                $payBooking->booking_status = 'Initiated'; //'Initiated';
				$payBooking->pay_order_id = 'BRIK-STAY-'.rand(10000000, 100000000);
                $payBooking->brik_unit_id = $request->brik_unit_id;
                $payBooking->client_id = $request->user()->id;
                $payBooking->save();

                foreach($request->payment_booking["booking_detail"] as $detail)
                {
                    $payBookingDetail = new BookingBookingDetail;
                    $payBookingDetail->booking_id = $payBooking->id;
                    $payBookingDetail->from_date = $detail["from"];
                    $payBookingDetail->to_date = $detail["to"];
                    $payBookingDetail->price = $detail["price"];
                    $payBookingDetail->nights = $detail["nights"];
                    $payBookingDetail->booking_price = $detail["booking_price"];
                    $payBookingDetail->save();
                }
                
                

                $payBooking = BookingBooking::find($payBooking->id);
                $has_payment_booking = true;
            }

            if($has_payment_booking){

                $returnurl = route('booking.payment.return').'?order='.$payBooking->pay_order_id.'&booking_request='.$booking_request->id;

                $payment = new BookingPayment();
                $payment->addClientInfo(Auth::user());
                $payment->addBookingInfo($payBooking); // If event is booking
                $payment->addReturnUrl($returnurl);
                $payment_req = $payment->generateSession();

                if($payment_req["cf_order_id"] && $payBooking->id){

                    $upd = BookingBooking::find($payBooking->id);
                    $upd->cf_order_id = $payment_req["cf_order_id"];
                    $upd->save();
                }
                
                DB::commit();

                return response()->json($this->throwSuccessResponse([
                    "mode" => 'payment',
                    "data" => $payment_req,
                ]));

            }else{

                if($coinBooking){
                    $this->coinDeduction($coinBooking);
                }

                DB::commit();

                $redirecturl = '/booking-status?request='.$booking_request->id;

                return response()->json($this->throwSuccessResponse([
                    "mode" => 'coin',
                    "data" => [
                        "redirecturl" => $redirecturl,
                    ],
                ]));

            }

            
        }catch(\Exception $e){

            DB::rollBack();
            return response()->json($this->throwFailedResponse($e->getMessage()));
        }

        

        
    }

    public function preCancel(Request $request)
    {
        $cancellations = [];

        try
        {
            $activeBookings = BookingBooking::where('booking_request_id', $request->booking_request_id)->get();

            foreach($activeBookings as $activeBooking)
            {
                    $ts_current_date = time();
                    $ts_date_check_in = strtotime($activeBooking->date_check_in);

                    if($ts_current_date > $ts_date_check_in)
                    {
                        return response()->json($this->throwFailedResponse('Can not cancel as Booking already in use.'));
                    }

                    $daycounts = $this->countDays(date('Y-m-d'), $activeBooking->date_check_in);

                    //dd($daycounts);

                    $percent_deduction = 0;

                    $ref = BookingCancellationPolicyMaster::where('days_from', '>=', $daycounts)->where('days_to','<=',$daycounts)->first();
                    
                    if($ref){
                        $percent_deduction = $ref->percent_deduction;
                    }

                    if($activeBooking->pay_by == 'Coin'){
                        
                        $coin_deducted = ($activeBooking->booking_cost_in_coin * ($percent_deduction / 100));

                        $cancellation_data = [
                            "booking_id" => $activeBooking->id,
                            "date_check_in" => $activeBooking->date_check_in,
                            "date_check_out" => $activeBooking->date_check_out,
                            "pay_by" => $activeBooking->pay_by,
                            "cancel_before_days" => $daycounts,
                            "percent_deduction" => $percent_deduction,
                            "booking_cost_in_coin" => $activeBooking->booking_cost_in_coin,
                            "coins_deduct" => $coin_deducted,
                            "coins_refund" => ($activeBooking->booking_cost_in_coin - $coin_deducted),
                        ];

                        $cancellations[] = $cancellation_data;

                    }else if($activeBooking->pay_by == 'Amount'){

                        $amount_deducted = ($activeBooking->net_payable_amount * ($percent_deduction / 100));
                        $cancellation_data = [
                            "booking_id" => $activeBooking->id,
                            "date_check_in" => $activeBooking->date_check_in,
                            "date_check_out" => $activeBooking->date_check_out,
                            "pay_by" => $activeBooking->pay_by,
                            "cancel_before_days" => $daycounts,
                            "percent_deduction" => $percent_deduction,
                            "net_payable_amount" => $activeBooking->net_payable_amount,
                            "amount_deduct" => $amount_deducted,
                            "amount_refund" => ($activeBooking->net_payable_amount - $amount_deducted),
                        ];

                        $cancellations[] = $cancellation_data;
                    }
            }

                return response()->json($this->throwSuccessResponse([
                    "data" => ["cancellation_records" => $cancellations]
                ]));

        }catch(\Exception $e){
            return response()->json($this->throwFailedResponse($e->getMessage()));
        }
    }

    public function cancel(Request $request){

        // $validator = Validator::make($request->all(), [
        //     'booking_id' => 'required',
        //     "pay_by" => 'required',
        //     "booking_cost_in_coin" => 'required_if:pay_by,Coin',
        //     "coins_deduct" => 'required_if:pay_by,Coin',
        //     "coins_refund" => 'required_if:pay_by,Coin',
        //     "net_payable_amount" => 'required_if:pay_by,Amount',
        //     "amount_deduct" => 'required_if:pay_by,Amount',
        //     "amount_refund" => 'required_if:pay_by,Amount',
        // ]);

        // if ($validator->fails()) {
        //     return response()->json($this->throwValidationFailedResponse($this->modifyValidationErrors($validator->messages()->get('*'))));
        // }

        try
        {
            foreach($request->cancellation_records as $cancel_record)
            {
                $booking = BookingBooking::find($cancel_record["booking_id"]);
                $booking_request = BookingBookingRequest::find($booking->booking_request_id);
            
                $booking->booking_status = 'Cancel';

                if($booking->pay_by == 'Coin'){
                    $booking->deducted_coins = $request->coins_deduct;
                    $booking->refunded_coins = $request->coins_refund;
                }
                if($booking->pay_by == 'Amount'){
                    $booking->deducted_amount = $request->amount_deduct;
                    $booking->refunded_amount = $request->amount_refund;
                }

                $booking->save();

                
                $booking_request->booking_status = 'Cancel';
                $booking_request->save();
            }

            return response()->json($this->throwSuccessResponse([
                "data" => null,
            ]));

        }catch(\Exception $e){
            return response()->json($this->throwFailedResponse($e->getMessage()));
        }

    }

    public function bookingHistory(Request $request){
		
        $booking_request_ids = BookingBooking::where('booking_status', 'Active')
        ->where('date_check_in','>', date("Y-m-d"))
        ->where('client_id', $request->user()->id)
        ->pluck('booking_request_id');

        $bookingRequests = BookingBookingRequest::with(['bookings','brik_unit'])
        ->whereIn('id', $booking_request_ids)->get();

      
        return response()->json($this->throwSuccessResponse([
            "data" => BookingRequestResource::collection($bookingRequests),
        ]));
    }

    public function bookingPastHistory(Request $request){
		
        $booking_request_ids = BookingBooking::where('booking_status', 'Active')
        ->where('date_check_in','<=', date("Y-m-d"))
        ->where('client_id', $request->user()->id)
        ->pluck('booking_request_id');

        $bookingRequests = BookingBookingRequest::with(['bookings','brik_unit'])
        ->whereIn('id', $booking_request_ids)->get();

      
        return response()->json($this->throwSuccessResponse([
            "data" => BookingRequestResource::collection($bookingRequests),
        ]));
    }

    public function bookingCancelHistory(Request $request){
		
        $booking_request_ids = BookingBooking::where('booking_status', 'Cancel')
        ->where('client_id', $request->user()->id)
        ->pluck('booking_request_id');

        $bookingRequests = BookingBookingRequest::with(['bookings','brik_unit'])
        ->whereIn('id', $booking_request_ids)->get();

      
        return response()->json($this->throwSuccessResponse([
            "data" => BookingRequestResource::collection($bookingRequests),
        ]));
    }

    // Payment Gateway Endpoint
    public function paymentreturn(Request $request)
    {
        $order_id = $request->order;
        $booking_request_id = $request->booking_request;
        
        $payment_status = 'ACTIVE';
		
        $payment = new BookingPayment();

        $payment_res = $payment->getOrderInfo($order_id);
		
		if($payment_res["order_status"]){
			
			
            if($payment_res["order_status"] == 'PAID'){
				
                BookingBookingRequest::where('id', $booking_request_id)->update([
                    "booking_status" => 'Active',
                ]);

                BookingBooking::where('booking_request_id', $booking_request_id)->update([
                    "booking_status" => 'Active',
                ]);
                
                $bookingDetail = BookingBooking::where('pay_order_id', $order_id)->first();
                BookingBooking::where('pay_order_id', $order_id)->update([
                    "cf_order_status" => $payment_res["order_status"],
                    "paid_amount" => $bookingDetail->net_payable_amount,
                    "balance_amount" => 0,
                ]);


                $coinBook = BookingBooking::where('booking_request_id', $booking_request_id)->where('pay_by','Coin')->first();
                if($coinBook){
                    $this->coinDeduction($coinBook);
                }

                 
            }else{

                    BookingBookingRequest::where('id', $booking_request_id)->update([
                        "booking_status" => 'Failed',
                    ]);
                    
                    BookingBooking::where('booking_request_id', $order_id)->update([
                        "booking_status" => 'Failed',
                        "cf_order_status" => $payment_res["order_status"],
                        
                    ]);

                
                
                
			}

        }

        $redirecturl = env("FRONTEND_APP_URL").'booking-status?request='.$booking_request_id;

        return Redirect::to($redirecturl);
    }

    // Payment Gateway Endpoint
    public function bookingOrderDetails(Request $request){

        $validator = Validator::make($request->all(), [
            'booking_request' => 'required',
        ]);

        if ($validator->fails()) {
            return response()->json($this->throwValidationFailedResponse($this->modifyValidationErrors($validator->messages()->get('*'))));
        }

        $bookings = BookingBooking::with(['brik_unit', 'client'])->where('booking_request_id', $request->booking_request)->get();

        // Add RU Booking API Here
       
        foreach($bookings as $booking){

            if($booking->brik_unit){

                if($booking->brik_unit->ru_property_id){

                    if(is_null($booking->ru_reservation_id)){

                        if($booking->pay_by == 'Amount'){
                            // Booking By Amount
                        
                            $ru =  new RU;
                            $ru->amountBooking($booking);
                            
                        }

                        if($booking->pay_by == 'Coin'){
                            // Booking By Coin
                            // if booking has not entered in rentals united
                            
                            $ru =  new RU;
                            $ru->coinBooking($booking);
                            
                        }
                    }
                }
            }
        }



        return response()->json($this->throwSuccessResponse(["data" => $bookings]));
    }

    protected function coinDeduction($booking)
    {

        $cointxn = new CoinWalletTransaction;
        $cointxn->booking_id = $booking->id;
        $cointxn->type = 'Debit';
        $cointxn->coin = $booking->booking_cost_in_coin;
        $cointxn->save();

        $client = Client::find($booking->client_id);
        $client->coins = ($client->coins - $booking->booking_cost_in_coin);
        $client->save();
        
    }

    protected function countDays($fromdate,$todate)
    {
        $from = strtotime($fromdate);
        $to = strtotime($todate);
        $diff = round(($to - $from) / (60 * 60 * 24));

        if($from === $to){
            return 1;
        }else{
            return (int)$diff;
        }
    }

    protected function countDaysCanBookUsingCoins(&$client, &$unit)
    {
        $per_night_coin = $unit->per_night_coin;
        $client_coins = $client->coins;

        if($client_coins > 0 && $per_night_coin > 0){
            return (int)floor($client_coins / $per_night_coin);
        }

        return 0;
        
    }

    protected function paymentBookingStucture($request, $checkin_date, $checkout_date, $night_counts)
    {
        $dynamicPrice = new BookingDynamicPrice($request->brik_unit_id, $request->date_check_in, $request->date_check_out);
        $dp = $dynamicPrice->extractDates()->groupDatesWithPrice()->finalGrouping()->addDaysAndPrice();
		//dd($dynamicPrice->extractDates()->groupDatesWithPrice()->finalGrouping());
        $booking_detail = $dp->get();
        // $booking_cost = $night_counts * $request->per_night_inr;
       $booking_cost = $dp->sum();
       $coupon_amt = $request->coupon_amount ? $request->coupon_amount : 0;
       $payable_amt = $booking_cost - $coupon_amt;
       $gst_amt = $payable_amt * (18/100);
       $net_payable_amt = $payable_amt + $gst_amt;

       return [
            "date_check_in" => $checkin_date,
            "date_check_out" => $checkout_date,
            //"price_per_night" => $request->per_night_inr,
            "booking_night_counts" => $night_counts,
            "booking_detail" => $booking_detail,
            "booking_cost_in_rupee" => $booking_cost,
            "brik_coupon_id" => $request->coupon_id,
            "coupon_amount" => $coupon_amt,
            "payable_amount" => $payable_amt,
            "gst_amount" => $gst_amt,
            "net_payable_amount" => $net_payable_amt,
        ];
    }

    protected function coinBookingStucture($checkin_date, $checkout_date, $night_counts, $per_night_coin)
    {
        return [
            "date_check_in" => $checkin_date,
            "date_check_out" => $checkout_date,
            "coins_per_night" => $per_night_coin,
            "booking_night_counts" => $night_counts,
            "booking_cost_in_coin" => ($per_night_coin * $night_counts),
       ];
    }

    protected function separateCoinAndPaymentDates($request, $days_client_book_using_coins, $booking_days)
    {
        $coin_pay_days = $days_client_book_using_coins;
        $payment_pay_days = $booking_days - $days_client_book_using_coins;

        $coin_checkin_date = $request->date_check_in;
        $coin_checkout_date = date('Y-m-d', strtotime($coin_checkin_date . ' +'.$coin_pay_days.' day'));

        $payment_checkin_date = $coin_checkout_date;
        $payment_checkout_date = $request->date_check_out;

        return [
            "coin_pay_days" => $coin_pay_days,
            "coin_checkin_date" => $coin_checkin_date,
            "coin_checkout_date" => $coin_checkout_date,
            "payment_pay_days" => $payment_pay_days,
            "payment_checkin_date" => $payment_checkin_date,
            "payment_checkout_date" => $payment_checkout_date,
        ];
    }

     

   

    
    // Testing Function will remove letter
    public function paymenttest(Request $request)
    {
        
        $client = Client::find(1);
        $booking = BookingBooking::find(1);

        $returnurl = route('booking.payment.return', $booking->order_id);

        $payment = new BookingPayment();

        $payment->addClientInfo($client);
        $payment->addBookingInfo($booking); // If event is booking
        $payment->addReturnUrl($returnurl);

        $payment_req = $payment->generateSession();
       
        $payment_session_id = $payment_req["payment_session_id"];

        $mode = $payment->mode;

        $booking_ids = [1];

        if($payment_req["status"] && $payment_req["cf_order_id"]){
           
            BookingBooking::whereIn('id', $booking_ids)->update(['cf_order_id' => $payment_req["cf_order_id"]]);
        }

        if($payment_req["status"]){
            return view("payment-test", compact('payment_session_id', 'mode'));
        }else{
            dd($payment_req);
        }
        
    }

    
}
